<?php

/**
 * This is the model class for table "erp_stock_ledger".
 *
 * The followings are the available columns in table 'erp_stock_ledger':
 * @property integer $id
 * @property string $transaction_date
 * @property string $transaction_type
 * @property string $type
 * @property integer $customer_vendor_id
 * @property integer $header_id
 * @property integer $item_id
 * @property integer $product_id
 * @property string $rate
 * @property integer $currency
 * @property string $exchange_rate
 * @property string $quantity
 * @property integer $unit
 * @property integer $default_product_unit
 * @property string $unit_exchange_rate
 * @property integer $created_by
 * @property integer $created_time
 * @property integer $outlet
 */
class StockLedger extends CActiveRecord {

    /**
     * @return string the associated database table name
     */
    public $amount;

    public function tableName() {
        return '{{stock_ledger}}';
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules() {
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
            array('transaction_type, customer_vendor_id, header_id, item_id, product_id, rate, currency, exchange_rate, quantity, unit, default_product_unit, unit_exchange_rate, created_by, created_time, brand_id', 'required'),
            array('customer_vendor_id, header_id, item_id, product_id, currency, unit, default_product_unit, created_by, brand_id,warehouse_id,rack_id,bin_id,batch_no,serial_no,hs_code', 'numerical', 'integerOnly' => true),
            array('transaction_type', 'length', 'max' => 3),
            array('type,bill_of_entry_exit', 'length', 'max' => 50),
            array('rate', 'length', 'max' => 10),
            array('exchange_rate, quantity, unit_exchange_rate,current_stock', 'numerical'),
            array('transaction_date', 'safe'),
            // The following rule is used by search().
            // @todo Please remove those attributes that should not be searched.
            array('id, transaction_date, transaction_type, type, customer_vendor_id, header_id, item_id, product_id, rate, currency, exchange_rate, quantity, unit, default_product_unit, unit_exchange_rate, created_by, created_time, brand_id', 'safe', 'on' => 'search'),
        );
    }

    /**
     * @return array relational rules.
     */
    public function relations() {
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        return array(
            'product_det' => array(self::BELONGS_TO, 'Inventory', 'product_id'),
            'unit_det' => array(self::BELONGS_TO, 'MasUnit', 'unit'),
            'pur_det' => array(self::BELONGS_TO, 'PurchaseInvoice', 'header_id', 'condition' => 'type ="PI"'),
            'purchasereturn_det' => array(self::BELONGS_TO, 'PurchaseReturn', 'header_id', 'condition' => 'type ="PR"'),
            'salesreturn_det' => array(self::BELONGS_TO, 'SalesReturn', 'header_id', 'condition' => 'type ="SR"'),
            'sales_det' => array(self::BELONGS_TO, 'SalesInvoice', 'header_id', 'condition' => 'type ="SI"'),
            'transaction_det' => array(self::BELONGS_TO, 'GoodsReceipt', 'header_id', 'condition' => 'type ="GR"'),
            'created_det' => array(self::BELONGS_TO, 'MasUsers', 'created_by'),
            'warehouse' => array(self::BELONGS_TO, 'MasInventoryWarehouse', 'warehouse_id'),
            'rack' => array(self::BELONGS_TO, 'MasInvRack', 'rack_id'),
            'cust_det' => array(self::BELONGS_TO, 'Customer', 'customer_vendor_id'),
            'bin' => array(self::BELONGS_TO, 'MasInvBin', 'bin_id'),
            
        );
    }

    /**
     * @return array customized attribute labels (name=>label)
     */
    public function attributeLabels() {
        return array(
            'id' => 'ID',
            'transaction_date' => 'Transaction Date',
            'transaction_type' => 'IN / OUT',
            'type' => 'Type',
            'customer_vendor_id' => 'Customer Vendor',
            'header_id' => 'Header',
            'item_id' => 'Item',
            'product_id' => 'Product',
            'rate' => 'Rate/Item',
            'currency' => 'Currency',
            'exchange_rate' => 'Exchange Rate',
            'quantity' => 'Quantity',
            'unit' => 'Unit',
            'default_product_unit' => 'Default Product Unit',
            'unit_exchange_rate' => 'Unit Exchange Rate',
            'created_by' => 'Created By',
            'created_time' => 'Created Time',
            'brand_id' => 'Brand',
        );
    }

    /**
     * Retrieves a list of models based on the current search/filter conditions.
     *
     * Typical usecase:
     * - Initialize the model fields with values from filter form.
     * - Execute this method to get CActiveDataProvider instance which will filter
     * models according to data in model fields.
     * - Pass data provider to CGridView, CListView or any similar widget.
     *
     * @return CActiveDataProvider the data provider that can return the models
     * based on the search/filter conditions.
     */
    public function search() {
        // @todo Please modify the following code to remove attributes that should not be searched.

        $criteria = new CDbCriteria;

        $criteria->compare('id', $this->id);
        $criteria->compare('transaction_date', $this->transaction_date, true);
        $criteria->compare('transaction_type', $this->transaction_type, true);
        $criteria->compare('type', $this->type, true);
        $criteria->compare('customer_vendor_id', $this->customer_vendor_id);
        $criteria->compare('header_id', $this->header_id);
        $criteria->compare('item_id', $this->item_id);
        $criteria->compare('product_id', $this->product_id);
        $criteria->compare('rate', $this->rate, true);
        $criteria->compare('currency', $this->currency);
        $criteria->compare('exchange_rate', $this->exchange_rate, true);
        $criteria->compare('quantity', $this->quantity, true);
        $criteria->compare('unit', $this->unit);
        $criteria->compare('default_product_unit', $this->default_product_unit);
        $criteria->compare('unit_exchange_rate', $this->unit_exchange_rate, true);
        $criteria->compare('created_by', $this->created_by);
        $criteria->compare('created_time', $this->created_time);
        $criteria->compare('brand_id', $this->brand_id);

        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
        ));
    }

    /**
     * Returns the static model of the specified AR class.
     * Please note that you should have this exact method in all your CActiveRecord descendants!
     * @param string $className active record class name.
     * @return StockLedger the static model class
     */
    public static function model($className = __CLASS__) {
        return parent::model($className);
    }

    public static function getAvgPurchaseRate($id, $date = '', $outlet_id = 0) {
        $condition = 'type IN ("PI","GR","MTM") AND transaction_type="IN" AND product_id= ' . $id;
        if ($date != '') {
            $condition.=' AND transaction_date <="' . $date . '"';
        }
        $model_purchase = self::model()->find(array('select' => 'IFNULL(SUM((rate*exchange_rate*quantity)),0) AS amount,IFNULL(SUM(unit_exchange_rate*quantity),0) AS quantity', 'condition' => $condition));
        $condition1 = 'type IN ("PR","MTM") AND transaction_type="OUT" AND product_id= ' . $id;
        if ($date != '') {
            $condition1.=' AND transaction_date <="' . $date . '"';
        }
        $model_purchase_return = self::model()->find(array('select' => 'IFNULL(SUM((rate*exchange_rate*quantity)),0) AS amount,IFNULL(SUM(unit_exchange_rate*quantity),0) AS quantity', 'condition' => $condition1));
        $purchase_amount = $model_purchase->quantity - $model_purchase_return->quantity == 0 ? 0 : ($model_purchase->amount - $model_purchase_return->amount) / ($model_purchase->quantity - $model_purchase_return->quantity);
        return $purchase_amount;
    }

    public function getType($type) {
        switch ($type) {
            case 'PI':
                $type = 'Purchase Invoice';
                break;
            case 'PR':
                $type = 'Purchase Return';
                break;
            case 'SI':
                $type = 'Sales Invoice';
                break;
            case 'SR':
                $type = 'Sales Return';
                break;
            case 'GI':
                $type = 'Goods Issue';
                break;
            case 'GR':
                $type = 'Goods Receipt';
                break;
            case 'GT':
                $type = 'Goods Transfer';
                break;
            case 'OB':
                $type = 'Opening Balance';
                break;
            case 'MTM':
                $type = 'Material To Material Conversion';
                break;
        }
        return $type;
    }

    public function getCustomerVendor($id) {
        $stockledger = self::model()->findByPK($id);
        $cus_ven = '-';
        if ($stockledger->type == 'PI' || $stockledger->type == 'PR') {
            $cus_ven = $stockledger->vendor_det->vendor_name;
        } else if ($stockledger->type == 'SI' || $stockledger->type == 'SR') {
            $cus_ven = $stockledger->customer_det->customer_name;
        }
        return $cus_ven;
    }
    
    public static function getPreviousCostPrice($product_id=0,$vendor_id=0){
		
	    $condition ='product_id='.$product_id.' AND transaction_type="IN"';
		if($vendor_id > 0){
		    $condition.='customer_vendor_id='.$vendor_id;
		}
	    $model = self::model()->find(array('select'=>'rate','condition'=>$condition,'order'=>'id DESC'));
	    return $model->rate;
	}
	public static function getPreviousSalesPrice($product_id=0,$customer_id=0){
		$condition ='product_id='.$product_id.' AND transaction_type="OUT"';
		if($customer_id > 0){
		    $condition.='customer_vendor_id='.$customer_id;
		}
	    $model = self::model()->find(array('select'=>'sales_price','condition'=>$condition,'order'=>'id DESC'));
	    return $model->sales_price;
	}

}
